if (typeof browser === "undefined") {
    globalThis.browser = chrome;
}

import {
    encodeRegister,
    decodeControlMessage,
    StreamReader,
} from "./codec.js";

import { handleCommandStream } from "./commands.js";

import { sendCurrentUrl, setupCapture } from "./capture.js";
import { DEFAULT_PROFILE, DEFAULT_CONFIG, SEVERITY_LEVELS } from "./defaults.js";

console.log("[Perch] Background script loading...");

let config = { ...DEFAULT_CONFIG };
let transport = null;
let session = null;
let controlSend = null;
let datagramWriter = null;
let reconnectAttempts = 0;
let reconnectTimeout = null;
let lastError = null;

async function loadConfig() {
    try {
        const stored = await browser.storage.local.get([
            "extensionName",
            "enabled",
            "severityFilter",
            "profiles",
            "activeProfile",
        ]);

        const storedProfiles = stored.profiles;
        const profiles =
            storedProfiles && storedProfiles.length > 0
                ? storedProfiles.map((p) => ({ ...DEFAULT_PROFILE, ...p }))
                : [{ ...DEFAULT_PROFILE }];

        config = {
            profiles,
            activeProfile: stored.activeProfile || "default",
            extensionName: stored.extensionName || DEFAULT_CONFIG.extensionName,
            enabled:
                stored.enabled !== undefined
                    ? stored.enabled
                    : DEFAULT_CONFIG.enabled,
            severityFilter:
                stored.severityFilter || DEFAULT_CONFIG.severityFilter,
        };
    } catch (e) {
        console.error("[Perch] Failed to load config:", e);
    }
}

function getActiveProfile() {
    return (
        config.profiles.find((p) => p.name === config.activeProfile) ||
        config.profiles[0] ||
        DEFAULT_PROFILE
    );
}

async function saveConfig(newConfig) {
    const oldActiveProfile = config.activeProfile;
    const oldProfile = getActiveProfile();
    const oldUrl = oldProfile.url;
    const oldUser = oldProfile.user;
    const oldPass = oldProfile.pass;

    config = { ...config, ...newConfig };
    await browser.storage.local.set(config);

    const newProfile = getActiveProfile();
    const profileSwitched =
        newConfig.activeProfile &&
        newConfig.activeProfile !== oldActiveProfile;
    const connectionChanged =
        newProfile.url !== oldUrl ||
        newProfile.user !== oldUser ||
        newProfile.pass !== oldPass;

    if (profileSwitched || connectionChanged) {
        disconnect();
        reconnectAttempts = 0;
        if (config.enabled) {
            connect();
        }
    }

    if (newConfig.enabled !== undefined) {
        if (newConfig.enabled && !isConnected()) {
            connect();
        } else if (!newConfig.enabled && transport) {
            disconnect();
        }
    }
}

async function updateBadge() {
    if (!isConnected()) {
        browser.action.setBadgeBackgroundColor({ color: "#ef4444" });
        browser.action.setBadgeText({ text: "!" });
        return;
    }

    const pinnedTabId = getPinnedTabId();
    if (!pinnedTabId) {
        browser.action.setBadgeBackgroundColor({ color: "#22c55e" });
        browser.action.setBadgeText({ text: "" });
        return;
    }

    try {
        const [activeTab] = await browser.tabs.query({
            active: true,
            currentWindow: true,
        });
        if (activeTab && activeTab.id === pinnedTabId) {
            browser.action.setBadgeBackgroundColor({ color: "#22c55e" });
            browser.action.setBadgeText({ text: "" });
        } else {
            browser.action.setBadgeBackgroundColor({ color: "#f97316" });
            browser.action.setBadgeText({ text: "○" });
        }
    } catch {
        browser.action.setBadgeBackgroundColor({ color: "#22c55e" });
        browser.action.setBadgeText({ text: "" });
    }
}

function getReconnectDelay() {
    const delays = [5000, 10000, 20000, 40000, 60000];
    return delays[Math.min(reconnectAttempts, delays.length - 1)];
}

function isConnected() {
    return !!(transport && session);
}

function shouldSendLog(level) {
    const filterLevel = SEVERITY_LEVELS[config.severityFilter] || 0;
    const logLevel = SEVERITY_LEVELS[level] || 0;
    return logLevel >= filterLevel;
}

function matchesPinnedTab(tabId, url) {
    const profile = getActiveProfile();
    if (!profile.pinnedTabId || !profile.pinnedOrigin) return true;
    if (tabId !== profile.pinnedTabId) return false;
    try {
        const urlOrigin = new URL(url).origin;
        return urlOrigin === profile.pinnedOrigin;
    } catch {
        return false;
    }
}

function getPinnedTabId() {
    const profile = getActiveProfile();
    return profile.pinnedTabId || null;
}

function buildUrl() {
    const profile = getActiveProfile();
    let url = profile.url;
    if (!url) return "";

    if (profile.user && profile.pass) {
        const auth = btoa(`${profile.user}:${profile.pass}`);
        const sep = url.includes("?") ? "&" : "?";
        url = `${url}${sep}auth=${encodeURIComponent(auth)}`;
    }

    return url;
}

function sendDatagram(data) {
    if (!datagramWriter) {
        console.log("[Perch] sendDatagram: no datagramWriter yet");
        return;
    }
    datagramWriter.write(data).catch((e) => {
        console.error("[Perch] sendDatagram error:", e);
        // Connection likely broken, trigger reconnect
        if (e.name === "InvalidStateError" || e.message?.includes("closed")) {
            console.log(
                "[Perch] Connection appears broken, triggering reconnect",
            );
            handleDisconnect();
        }
    });
}

async function connect() {
    if (!config.enabled) return;
    if (transport) return;

    const url = buildUrl();
    if (!url) {
        lastError = "No URL configured";
        updateBadge();
        return;
    }
    console.log("[Perch] Connecting to", url);

    try {
        transport = new WebTransport(url);

        await transport.ready;
        console.log("[Perch] WebTransport ready");

        session = transport;

        const controlStream = await session.createBidirectionalStream();
        controlSend = controlStream.writable.getWriter();
        const controlRecv = controlStream.readable.getReader();

        const registerMsg = encodeRegister(
            config.extensionName,
            navigator.userAgent,
        );
        await controlSend.write(registerMsg);

        const reader = new StreamReader({ read: () => controlRecv.read() });
        const response = await decodeControlMessage(reader);

        if (response.type === "register_error") {
            lastError = response.message;
            console.error("[Perch] Registration failed:", response.message);
            updateBadge();
            transport.close();
            transport = null;
            session = null;
            scheduleReconnect();
            return;
        }

        console.log("[Perch] Registered successfully");
        reconnectAttempts = 0;
        lastError = null;
        updateBadge();

        datagramWriter = session.datagrams.writable.getWriter();

        sendCurrentUrl(sendDatagram);
        listenForCommands();

        transport.closed
            .then(() => {
                console.log("[Perch] Transport closed");
                handleDisconnect();
            })
            .catch((e) => {
                console.error("[Perch] Transport error:", e);
                lastError = e.message || "Connection error";
                handleDisconnect();
            });
    } catch (e) {
        const errorMsg = e.message || e.toString();
        const errorSource = e.source || "";
        const errorCode = e.streamErrorCode ?? e.sessionErrorCode ?? "";
        console.error(
            "[Perch] Connection failed:",
            errorMsg,
            errorSource ? `(source: ${errorSource})` : "",
            errorCode ? `(code: ${errorCode})` : "",
        );
        lastError = errorMsg || "Connection failed";
        updateBadge();
        transport = null;
        session = null;
        scheduleReconnect();
    }
}

function handleDisconnect() {
    transport = null;
    session = null;
    controlSend = null;
    datagramWriter = null;
    updateBadge();
    if (config.enabled) {
        scheduleReconnect();
    }
}

function disconnect() {
    if (reconnectTimeout) {
        clearTimeout(reconnectTimeout);
        reconnectTimeout = null;
    }

    if (transport) {
        transport.close();
        transport = null;
        session = null;
        controlSend = null;
        datagramWriter = null;
    }

    updateBadge();
}

function scheduleReconnect() {
    if (reconnectTimeout) return;

    const delay = getReconnectDelay();
    console.log(`[Perch] Reconnecting in ${delay / 1000}s...`);

    reconnectTimeout = setTimeout(() => {
        reconnectTimeout = null;
        reconnectAttempts++;
        connect();
    }, delay);
}

const commandCtx = {
    sendDatagram,
    getPinnedTabId,
    getSession: () => session,
};

async function listenForCommands() {
    if (!session) return;

    const reader = session.incomingBidirectionalStreams.getReader();

    try {
        while (true) {
            const { value: stream, done } = await reader.read();
            if (done) break;

            handleCommandStream(stream, commandCtx);
        }
    } catch (e) {
        console.error("[Perch] Error listening for commands:", e);
    }
}

setupCapture({
    getConfig: () => config,
    sendDatagram,
    isConnected,
    shouldSendLog,
    matchesPinnedTab,
    saveConfig,
    disconnect,
    connect,
    updateBadge,
    resetReconnect: () => { reconnectAttempts = 0; },
    getStatus: () => ({
        connected: isConnected(),
        connecting: transport && !session,
        error: lastError,
        config: config,
    }),
});

loadConfig().then(() => {
    if (config.enabled) {
        connect();
    } else {
        updateBadge();
    }
});
