// Perch Browser Observer - Command Handlers
// Handles commands received from the Perch server via WebTransport bidi streams

import {
    encodeCommandResult,
    encodeScreenshotData,
    encodePongDatagram,
    decodeCommand,
    StreamReader,
} from "./codec.js";

// Send command result on the bidi stream
async function sendCommandResult(writer, id, success, result, error) {
    const data = encodeCommandResult(id, success, result, error);
    await writer.write(data);
}

// Handle a command stream from server
// ctx: { sendDatagram, getPinnedTabId, getSession }
export async function handleCommandStream(stream, ctx) {
    const recv = stream.readable.getReader();
    const send = stream.writable.getWriter();
    const reader = new StreamReader({ read: () => recv.read() });

    try {
        const command = await decodeCommand(reader);
        await handleCommand(command, send, ctx);
    } catch (e) {
        console.error("[Perch] Error handling command:", e);
    } finally {
        try {
            await send.close();
        } catch {}
        try {
            recv.releaseLock();
        } catch {}
    }
}

// Handle a command from server
async function handleCommand(command, sendWriter, ctx) {
    switch (command.type) {
        case "ping":
            ctx.sendDatagram(encodePongDatagram());
            break;

        case "screenshot":
            await takeScreenshot(command.id, ctx.getSession);
            break;

        case "execute_js":
            await executeInTab(command.id, command.code, sendWriter, ctx.getPinnedTabId);
            break;

        case "click":
            await clickInTab(command.id, command.selector, sendWriter, ctx.getPinnedTabId);
            break;

        case "fill":
            await fillInTab(
                command.id,
                command.selector,
                command.value,
                sendWriter,
                ctx.getPinnedTabId,
            );
            break;

        case "get_text":
            await getTextInTab(command.id, command.selector, sendWriter, ctx.getPinnedTabId);
            break;

        case "navigate":
            await navigateTab(command.id, command.url, sendWriter, ctx.getPinnedTabId);
            break;

        default:
            console.log("[Perch] Unknown command:", command.type);
    }
}

// Navigate pinned tab to URL
async function navigateTab(id, url, sendWriter, getPinnedTabId) {
    const tabId = getPinnedTabId();
    if (!tabId) {
        await sendCommandResult(sendWriter, id, false, null, "No tab pinned");
        return;
    }

    try {
        await browser.tabs.update(tabId, { url });
        await sendCommandResult(sendWriter, id, true, "ok", null);
    } catch (e) {
        await sendCommandResult(sendWriter, id, false, null, e.message);
    }
}

// Execute JavaScript in pinned tab
async function executeInTab(id, code, sendWriter, getPinnedTabId) {
    const tabId = getPinnedTabId();
    if (!tabId) {
        await sendCommandResult(sendWriter, id, false, null, "No tab pinned");
        return;
    }

    try {
        const results = await browser.scripting.executeScript({
            target: { tabId },
            func: (code) => {
                try {
                    const result = eval(code);
                    return {
                        success: true,
                        result: result === undefined ? null : String(result),
                    };
                } catch (e) {
                    return { success: false, error: e.message };
                }
            },
            args: [code],
        });

        const result = results[0]?.result;
        if (result?.success) {
            await sendCommandResult(sendWriter, id, true, result.result, null);
        } else {
            await sendCommandResult(
                sendWriter,
                id,
                false,
                null,
                result?.error || "Unknown error",
            );
        }
    } catch (e) {
        await sendCommandResult(sendWriter, id, false, null, e.message);
    }
}

// Click element in pinned tab
async function clickInTab(id, selector, sendWriter, getPinnedTabId) {
    const tabId = getPinnedTabId();
    if (!tabId) {
        await sendCommandResult(sendWriter, id, false, null, "No tab pinned");
        return;
    }

    try {
        const results = await browser.scripting.executeScript({
            target: { tabId },
            func: (selector) => {
                const el = document.querySelector(selector);
                if (!el)
                    return {
                        success: false,
                        error: "Element not found: " + selector,
                    };
                el.click();
                return { success: true };
            },
            args: [selector],
        });

        const result = results[0]?.result;
        if (result?.success) {
            await sendCommandResult(sendWriter, id, true, null, null);
        } else {
            await sendCommandResult(
                sendWriter,
                id,
                false,
                null,
                result?.error || "Click failed",
            );
        }
    } catch (e) {
        await sendCommandResult(sendWriter, id, false, null, e.message);
    }
}

// Fill input in pinned tab
async function fillInTab(id, selector, value, sendWriter, getPinnedTabId) {
    const tabId = getPinnedTabId();
    if (!tabId) {
        await sendCommandResult(sendWriter, id, false, null, "No tab pinned");
        return;
    }

    try {
        const results = await browser.scripting.executeScript({
            target: { tabId },
            func: (selector, value) => {
                const el = document.querySelector(selector);
                if (!el)
                    return {
                        success: false,
                        error: "Element not found: " + selector,
                    };
                el.value = value;
                el.dispatchEvent(new Event("input", { bubbles: true }));
                el.dispatchEvent(new Event("change", { bubbles: true }));
                return { success: true };
            },
            args: [selector, value],
        });

        const result = results[0]?.result;
        if (result?.success) {
            await sendCommandResult(sendWriter, id, true, null, null);
        } else {
            await sendCommandResult(
                sendWriter,
                id,
                false,
                null,
                result?.error || "Fill failed",
            );
        }
    } catch (e) {
        await sendCommandResult(sendWriter, id, false, null, e.message);
    }
}

// Get text from element in pinned tab
async function getTextInTab(id, selector, sendWriter, getPinnedTabId) {
    const tabId = getPinnedTabId();
    if (!tabId) {
        await sendCommandResult(sendWriter, id, false, null, "No tab pinned");
        return;
    }

    try {
        const results = await browser.scripting.executeScript({
            target: { tabId },
            func: (selector) => {
                const el = document.querySelector(selector);
                if (!el)
                    return {
                        success: false,
                        error: "Element not found: " + selector,
                    };
                return { success: true, result: el.textContent };
            },
            args: [selector],
        });

        const result = results[0]?.result;
        if (result?.success) {
            await sendCommandResult(sendWriter, id, true, result.result, null);
        } else {
            await sendCommandResult(
                sendWriter,
                id,
                false,
                null,
                result?.error || "Get text failed",
            );
        }
    } catch (e) {
        await sendCommandResult(sendWriter, id, false, null, e.message);
    }
}

// Take screenshot and send via uni stream
async function takeScreenshot(id, getSession) {
    const session = getSession();
    if (!session) return;

    try {
        const [tab] = await browser.tabs.query({
            active: true,
            currentWindow: true,
        });
        if (!tab) {
            console.error("[Perch] No active tab for screenshot");
            return;
        }

        const dataUrl = await browser.tabs.captureVisibleTab(null, {
            format: "jpeg",
            quality: 80,
        });
        const resizedData = await resizeImage(dataUrl, 1024);
        const base64Data = resizedData.replace(/^data:image\/\w+;base64,/, "");

        // Decode base64 to bytes
        const binaryStr = atob(base64Data);
        const bytes = new Uint8Array(binaryStr.length);
        for (let i = 0; i < binaryStr.length; i++) {
            bytes[i] = binaryStr.charCodeAt(i);
        }

        // Send via uni stream
        const stream = await session.createUnidirectionalStream();
        const writer = stream.getWriter();
        const data = encodeScreenshotData(id, bytes, tab.url, tab.title);
        await writer.write(data);
        await writer.close();
    } catch (e) {
        console.error("[Perch] Screenshot failed:", e);
    }
}

// Resize image
async function resizeImage(dataUrl, maxWidth) {
    return new Promise((resolve, reject) => {
        const img = new Image();
        img.onload = () => {
            let width = img.width;
            let height = img.height;

            if (width > maxWidth) {
                const ratio = maxWidth / width;
                width = maxWidth;
                height = Math.round(height * ratio);
            }

            let canvas;
            if (typeof OffscreenCanvas !== "undefined") {
                canvas = new OffscreenCanvas(width, height);
            } else {
                canvas = document.createElement("canvas");
                canvas.width = width;
                canvas.height = height;
            }

            const ctx = canvas.getContext("2d");
            ctx.drawImage(img, 0, 0, width, height);

            if (canvas.convertToBlob) {
                canvas
                    .convertToBlob({ type: "image/jpeg", quality: 0.8 })
                    .then((blob) => {
                        const reader = new FileReader();
                        reader.onload = () => resolve(reader.result);
                        reader.onerror = reject;
                        reader.readAsDataURL(blob);
                    })
                    .catch(reject);
            } else {
                resolve(canvas.toDataURL("image/jpeg", 0.8));
            }
        };
        img.onerror = reject;
        img.src = dataUrl;
    });
}
