// Perch Console Interceptor - runs in page's main world
(function() {
  if (window.__perchInjected__) return;
  window.__perchInjected__ = true;

  const MAX_MESSAGE_SIZE = 5 * 1024;
  const MAX_LOGS_PER_SECOND = 100;

  let logCount = 0;
  let lastResetTime = Date.now();
  let droppedCount = 0;

  function checkRateLimit() {
    const now = Date.now();
    if (now - lastResetTime >= 1000) {
      if (droppedCount > 0) {
        window.postMessage({
          __perch__: true,
          type: 'log',
          level: 'warn',
          message: '[Perch] Rate limit: dropped ' + droppedCount + ' logs',
          url: window.location.href,
          timestamp: now
        }, '*');
        droppedCount = 0;
      }
      logCount = 0;
      lastResetTime = now;
    }
    if (logCount >= MAX_LOGS_PER_SECOND) {
      droppedCount++;
      return false;
    }
    logCount++;
    return true;
  }

  function truncate(msg) {
    if (msg.length > MAX_MESSAGE_SIZE) {
      return msg.substring(0, MAX_MESSAGE_SIZE - 15) + '...[truncated]';
    }
    return msg;
  }

  function argsToString(args) {
    return Array.from(args).map(function(arg) {
      if (arg === null) return 'null';
      if (arg === undefined) return 'undefined';
      if (typeof arg === 'string') return arg;
      if (typeof arg === 'number' || typeof arg === 'boolean') return String(arg);
      if (arg instanceof Error) {
        return arg.name + ': ' + arg.message + (arg.stack ? '\n' + arg.stack : '');
      }
      try {
        return JSON.stringify(arg, null, 2);
      } catch (e) {
        return String(arg);
      }
    }).join(' ');
  }

  function getSourceLocation() {
    const stack = new Error().stack;
    if (!stack) return { file: '', line: 0 };
    const lines = stack.split('\n');
    for (let i = 3; i < lines.length; i++) {
      const line = lines[i];
      const match = line.match(/(?:at\s+)?(?:.*?\s+)?(?:\()?([^()]+):(\d+)(?::\d+)?(?:\))?$/);
      if (match && !line.includes('perch')) {
        return { file: match[1], line: parseInt(match[2], 10) };
      }
    }
    return { file: '', line: 0 };
  }

  const levelMap = { debug: 'debug', log: 'info', info: 'info', warn: 'warn', error: 'error' };
  const original = {};

  ['debug', 'log', 'info', 'warn', 'error'].forEach(function(level) {
    original[level] = console[level];
    console[level] = function() {
      original[level].apply(console, arguments);
      if (!checkRateLimit()) return;
      const loc = getSourceLocation();
      window.postMessage({
        __perch__: true,
        type: 'log',
        level: levelMap[level],
        message: truncate(argsToString(arguments)),
        url: window.location.href,
        timestamp: Date.now(),
        source_file: loc.file,
        line_number: loc.line
      }, '*');
    };
  });

  window.addEventListener('error', function(e) {
    if (!checkRateLimit()) return;
    window.postMessage({
      __perch__: true,
      type: 'log',
      level: 'error',
      message: truncate(e.message + (e.error && e.error.stack ? '\n' + e.error.stack : '')),
      url: window.location.href,
      timestamp: Date.now(),
      source_file: e.filename || '',
      line_number: e.lineno || 0
    }, '*');
  });

  window.addEventListener('unhandledrejection', function(e) {
    if (!checkRateLimit()) return;
    let msg = 'Unhandled Promise Rejection: ';
    if (e.reason instanceof Error) {
      msg += e.reason.message + (e.reason.stack ? '\n' + e.reason.stack : '');
    } else {
      try { msg += JSON.stringify(e.reason); } catch(x) { msg += String(e.reason); }
    }
    window.postMessage({
      __perch__: true,
      type: 'log',
      level: 'error',
      message: truncate(msg),
      url: window.location.href,
      timestamp: Date.now(),
      source_file: '',
      line_number: 0
    }, '*');
  });
})();
