// Perch Browser Observer - Popup Script
// Multi-profile configuration

// Chrome <144 compatibility: alias chrome to browser
if (typeof browser === 'undefined') {
  globalThis.browser = chrome;
}

document.addEventListener('DOMContentLoaded', async () => {
  const enabledCheckbox = document.getElementById('enabled');
  const perchUrlInput = document.getElementById('perchUrl');
  const tabSelect = document.getElementById('tabSelect');
  const pinCurrentBtn = document.getElementById('pinCurrentBtn');
  const authUserInput = document.getElementById('authUser');
  const authPassInput = document.getElementById('authPass');
  const extensionNameInput = document.getElementById('extensionName');
  const severityFilterSelect = document.getElementById('severityFilter');
  const exportBtn = document.getElementById('exportBtn');
  const importBtn = document.getElementById('importBtn');
  const importFile = document.getElementById('importFile');
  const popoutBtn = document.getElementById('popoutBtn');
  const statusDiv = document.getElementById('status');
  const statusText = statusDiv.querySelector('.status-text');
  const profileSelect = document.getElementById('profileSelect');
  const addProfileBtn = document.getElementById('addProfileBtn');
  const removeProfileBtn = document.getElementById('removeProfileBtn');

  let config = null;
  let saveTimeout = null;

  // Pop out to detached window
  popoutBtn.addEventListener('click', () => {
    browser.windows.create({
      url: browser.runtime.getURL('popup/popup.html'),
      type: 'popup',
      width: 360,
      height: 480
    });
    window.close();
  });

  // Update connection status indicator
  async function updateStatus() {
    try {
      const response = await browser.runtime.sendMessage({ type: 'getStatus' });
      if (!response) return;

      if (response.connected) {
        statusDiv.className = 'status connected';
        statusText.textContent = 'Connected';
      } else if (response.connecting) {
        statusDiv.className = 'status connecting';
        statusText.textContent = 'Connecting...';
      } else if (response.error) {
        statusDiv.className = 'status error';
        statusText.textContent = response.error;
      } else if (!response.config?.enabled) {
        statusDiv.className = 'status disabled';
        statusText.textContent = 'Disabled';
      } else {
        statusDiv.className = 'status disconnected';
        statusText.textContent = 'Disconnected';
      }
    } catch (e) {
      // Background script not ready yet
    }
  }

  // Get the active profile
  function getProfile() {
    if (!config?.profiles?.length) {
      return { url: '', user: '', pass: '', pinnedTabId: null, pinnedOrigin: null };
    }
    return config.profiles.find(p => p.name === config.activeProfile) || config.profiles[0];
  }

  // Populate profile selector dropdown
  function populateProfileSelect() {
    if (!config) return;
    profileSelect.innerHTML = '';
    for (const profile of config.profiles) {
      const option = document.createElement('option');
      option.value = profile.name;
      option.textContent = profile.name;
      if (profile.name === config.activeProfile) {
        option.selected = true;
      }
      profileSelect.appendChild(option);
    }
    removeProfileBtn.disabled = config.profiles.length <= 1;
  }

  // Fill form fields from active profile
  function fillFormFromProfile() {
    const profile = getProfile();
    perchUrlInput.value = profile.url || '';
    authUserInput.value = profile.user || '';
    authPassInput.value = profile.pass || '';
    populateTabSelect();
  }

  // Populate tab dropdown
  async function populateTabSelect() {
    const profile = getProfile();
    const tabs = await browser.tabs.query({});

    tabSelect.innerHTML = '';

    let hasSelection = false;
    for (const tab of tabs) {
      if (!tab.url || tab.url.startsWith('about:') || tab.url.startsWith('moz-extension:')) {
        continue;
      }
      const option = document.createElement('option');
      option.value = `${tab.id}|${new URL(tab.url).origin}`;

      const title = tab.title || tab.url;
      const displayTitle = title.length > 40 ? title.substring(0, 40) + '...' : title;
      option.textContent = displayTitle;

      if (profile.pinnedTabId === tab.id) {
        option.selected = true;
        hasSelection = true;
      }

      tabSelect.appendChild(option);
    }

    if (!hasSelection) {
      const placeholder = document.createElement('option');
      placeholder.value = '';
      placeholder.textContent = 'All tabs';
      placeholder.selected = true;
      tabSelect.insertBefore(placeholder, tabSelect.firstChild);
    }
  }

  // Load config (with retry)
  async function loadConfig(retries = 3) {
    try {
      const response = await browser.runtime.sendMessage({ type: 'getStatus' });
      if (!response) throw new Error('No response');
      config = response.config;

      enabledCheckbox.checked = config.enabled;
      extensionNameInput.value = config.extensionName || '';
      severityFilterSelect.value = config.severityFilter || 'all';

      populateProfileSelect();
      fillFormFromProfile();
      updateStatus();
    } catch (e) {
      if (retries > 0) {
        await new Promise(r => setTimeout(r, 300));
        return loadConfig(retries - 1);
      }
      console.error('Failed to load config:', e);
      statusDiv.className = 'status error';
      statusText.textContent = 'Loading...';
    }
  }

  // Auto-save (debounced)
  function scheduleAutoSave() {
    if (saveTimeout) clearTimeout(saveTimeout);
    saveTimeout = setTimeout(autoSave, 500);
  }

  // Save config
  async function autoSave() {
    if (!config) return;

    const profile = getProfile();
    profile.url = perchUrlInput.value.trim();
    profile.user = authUserInput.value.trim();
    profile.pass = authPassInput.value;

    config.extensionName = extensionNameInput.value.trim();
    config.severityFilter = severityFilterSelect.value;

    try {
      await browser.runtime.sendMessage({ type: 'saveConfig', config });
      setTimeout(updateStatus, 300);
    } catch (e) {
      console.error('Failed to save config:', e);
    }
  }

  // Export settings
  function exportSettings() {
    if (!config) return;
    const data = JSON.stringify(config, null, 2);
    const blob = new Blob([data], { type: 'application/json' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = 'perch-settings.json';
    a.click();
    URL.revokeObjectURL(url);
  }

  // Import settings
  async function importSettings(file) {
    try {
      const text = await file.text();
      const imported = JSON.parse(text);

      if (!imported.profiles || !Array.isArray(imported.profiles)) {
        alert('Invalid settings file');
        return;
      }

      config = imported;
      await browser.runtime.sendMessage({ type: 'saveConfig', config });

      enabledCheckbox.checked = config.enabled;
      extensionNameInput.value = config.extensionName || '';
      severityFilterSelect.value = config.severityFilter || 'all';

      populateProfileSelect();
      fillFormFromProfile();
      setTimeout(updateStatus, 500);
    } catch (e) {
      console.error('Import failed:', e);
      alert('Failed to import settings');
    }
  }

  // Profile management
  profileSelect.addEventListener('change', async () => {
    if (!config) return;
    config.activeProfile = profileSelect.value;
    fillFormFromProfile();
    try {
      await browser.runtime.sendMessage({ type: 'saveConfig', config });
      setTimeout(updateStatus, 500);
    } catch (e) {
      console.error('Failed to switch profile:', e);
    }
  });

  addProfileBtn.addEventListener('click', async () => {
    if (!config) return;
    const name = prompt('Profile name:');
    if (!name || !name.trim()) return;
    const trimmed = name.trim();
    if (config.profiles.some(p => p.name === trimmed)) {
      alert('A profile with that name already exists');
      return;
    }
    config.profiles.push({ name: trimmed, url: '', user: '', pass: '' });
    config.activeProfile = trimmed;
    populateProfileSelect();
    fillFormFromProfile();
    try {
      await browser.runtime.sendMessage({ type: 'saveConfig', config });
      setTimeout(updateStatus, 500);
    } catch (e) {
      console.error('Failed to add profile:', e);
    }
  });

  removeProfileBtn.addEventListener('click', async () => {
    if (!config || config.profiles.length <= 1) return;
    const idx = config.profiles.findIndex(p => p.name === config.activeProfile);
    if (idx === -1) return;
    config.profiles.splice(idx, 1);
    config.activeProfile = config.profiles[0].name;
    populateProfileSelect();
    fillFormFromProfile();
    try {
      await browser.runtime.sendMessage({ type: 'saveConfig', config });
      setTimeout(updateStatus, 500);
    } catch (e) {
      console.error('Failed to remove profile:', e);
    }
  });

  // Event listeners
  exportBtn.addEventListener('click', exportSettings);
  importBtn.addEventListener('click', () => importFile.click());
  importFile.addEventListener('change', (e) => {
    if (e.target.files[0]) {
      importSettings(e.target.files[0]);
      e.target.value = '';
    }
  });

  enabledCheckbox.addEventListener('change', async () => {
    if (!config) return;
    config.enabled = enabledCheckbox.checked;
    await browser.runtime.sendMessage({
      type: 'saveConfig',
      config: { enabled: enabledCheckbox.checked }
    });
    setTimeout(updateStatus, 500);
  });

  tabSelect.addEventListener('change', async () => {
    if (!config) return;
    const value = tabSelect.value;
    const profile = getProfile();

    if (value) {
      const [tabId, origin] = value.split('|');
      profile.pinnedTabId = parseInt(tabId, 10);
      profile.pinnedOrigin = origin;
      try {
        await browser.scripting.executeScript({
          target: { tabId: profile.pinnedTabId, allFrames: true },
          files: ['content.js']
        });
      } catch (e) {
        console.log('Content script injection skipped:', e.message);
      }
    } else {
      profile.pinnedTabId = null;
      profile.pinnedOrigin = null;
    }
    await browser.runtime.sendMessage({ type: 'saveConfig', config });
  });

  pinCurrentBtn.addEventListener('click', async () => {
    if (!config) return;
    const [tab] = await browser.tabs.query({ active: true, currentWindow: true });
    if (!tab || !tab.url) return;

    const origin = new URL(tab.url).origin;
    const value = `${tab.id}|${origin}`;

    for (const option of tabSelect.options) {
      if (option.value === value) {
        option.selected = true;
        break;
      }
    }

    const profile = getProfile();
    profile.pinnedTabId = tab.id;
    profile.pinnedOrigin = origin;

    try {
      await browser.scripting.executeScript({
        target: { tabId: tab.id, allFrames: true },
        files: ['content.js']
      });
    } catch (e) {
      console.log('Content script injection skipped:', e.message);
    }

    await browser.runtime.sendMessage({ type: 'saveConfig', config });
  });

  perchUrlInput.addEventListener('input', scheduleAutoSave);
  authUserInput.addEventListener('input', scheduleAutoSave);
  authPassInput.addEventListener('input', scheduleAutoSave);
  extensionNameInput.addEventListener('input', scheduleAutoSave);
  severityFilterSelect.addEventListener('change', scheduleAutoSave);

  // Initialize
  loadConfig();
  setInterval(updateStatus, 2000);
});
